--
-- HUD
--


HUDSetting = {}

function HUDSetting:new()
	self.active = true
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	return self
end

function HUDSetting:onStateChange(state, checkboxElement, loadFromSavegame)
	g_currentMission.hud:setIsVisible(state)

	if checkboxElement ~= nil then
		local target = checkboxElement.target

		if target ~= nil then
			target.multiCrosshair:setDisabled(not state or g_noHudModeEnabled)
			target.multiDate:setDisabled(not state)

			local disabled = not state or g_additionalSettingsManager:getSettingStateByName("date") == 0

			target.multiClockPosition:setDisabled(disabled)
			target.checkClockBackground:setDisabled(disabled)
			target.checkHourFormat:setDisabled(not state)
			target.buttonDateColor:setDisabled(disabled)
		end
	end
end

function HUDSetting:onFrameOpen(checkboxElement)
	self.active = g_currentMission.hud:getIsVisible()
end


--
-- Crosshair
--


CrosshairSetting = {}

function CrosshairSetting:new()
	AdditionalSettingsUtil.registerEventListener("onMissionStarted", self)

	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	return self
end

function CrosshairSetting:onMissionStarted()
	local player = g_currentMission.player

	if player ~= nil then
		AdditionalSettingsUtil.prependedFunction(player.petOverlay, "render", self, "render")
		AdditionalSettingsUtil.prependedFunction(player.brushOverlay, "render", self, "render")
		AdditionalSettingsUtil.prependedFunction(player.pickedUpObjectOverlay, "render", self, "render")
		AdditionalSettingsUtil.prependedFunction(player.aimOverlay, "render", self, "renderAim")
	end
end

function CrosshairSetting:onFrameOpen(optionElement)
	optionElement:setDisabled(not g_currentMission.hud:getIsVisible() or g_noHudModeEnabled)
end

function CrosshairSetting.render(setting, object)
	object:setIsVisible(setting.state ~= 2)
end

function CrosshairSetting.renderAim(setting, object)
	object:setIsVisible(setting.state == 0 or setting.state == 1 and setting.getIsICActive())
end

function CrosshairSetting.getIsICActive()
	if g_modIsLoaded.FS22_simpleIC then
		for _, displayActionEvent in ipairs(g_inputBinding:getDisplayActionEvents()) do
			if displayActionEvent.action.name == "INTERACT_IC_ONFOOT" then
				return true
			end
		end
	end

	return false
end


--
-- Date
--


DateSetting = {}

function DateSetting:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	self.dateFormat = ""

	self.blinking = true
	self.blinkTime = 0
	self.blinkSec = -1

	local playerLeftX = g_safeFrameOffsetX
	local playerRightX = 1 - g_safeFrameOffsetX

	local playerTopY = 1 - g_safeFrameOffsetY / 2
	local playerBottomY = g_safeFrameOffsetY / 2

	local gearElementOverlay = g_currentMission.hud.speedMeter.gearElement.overlay
	local vehicleRightX = gearElementOverlay.x + gearElementOverlay.width

	local defaultBgColorDark = {0, 0, 0, 0.75}
	local defaultBgColorLight = {0.01, 0.01, 0.01, 0.6}

	self.clockSettings = {
		[0] = {
			horizontalAlignment = RenderText.ALIGN_RIGHT,
			verticalAligment = RenderText.VERTICAL_ALIGN_MIDDLE,
			playerBgColor = defaultBgColorDark,
			playerPosition = {
				posX = playerRightX,
				posY = playerTopY
			}
		},
		[1] = {
			horizontalAlignment = RenderText.ALIGN_RIGHT,
			verticalAligment = RenderText.VERTICAL_ALIGN_MIDDLE,
			playerBgColor = defaultBgColorDark,
			vehicleBgColor = defaultBgColorLight,
			playerPosition = {
				posX = playerRightX,
				posY = playerBottomY
			},
			vehiclePosition = {
				posX = vehicleRightX,
				posY = playerBottomY
			}
		},
		[2] = {
			horizontalAlignment = RenderText.ALIGN_LEFT,
			verticalAligment = RenderText.VERTICAL_ALIGN_MIDDLE,
			playerBgColor = defaultBgColorLight,
			playerPosition = {
				posX = playerLeftX,
				posY = playerTopY
			}
		},
		[3] = {
			horizontalAlignment = RenderText.ALIGN_LEFT,
			verticalAligment = RenderText.VERTICAL_ALIGN_MIDDLE,
			playerBgColor = defaultBgColorDark,
			playerPosition = {
				posX = playerLeftX,
				posY = playerBottomY
			}
		}
	}

	local function getDateText(text)
		if g_i18n:hasText("additionalSettings_" .. text) then
			return g_i18n:getText("additionalSettings_" .. text)
		end

		return g_i18n:getText(text)
	end

	self.dayName = {
		["Monday"] = getDateText("ui_financesDay1"),
		["Tuesday"] = getDateText("ui_financesDay2"),
		["Wednesday"] = getDateText("ui_financesDay3"),
		["Thursday"] = getDateText("ui_financesDay4"),
		["Friday"] = getDateText("ui_financesDay5"),
		["Saturday"] = getDateText("ui_financesDay6"),
		["Sunday"] = getDateText("ui_financesDay7")
	}

	self.monthName = {
		["January"] = getDateText("ui_month1"),
		["February"] = getDateText("ui_month2"),
		["March"] = getDateText("ui_month3"),
		["April"] = getDateText("ui_month4"),
		["May"] = getDateText("ui_month5"),
		["June"] = getDateText("ui_month6"),
		["July"] = getDateText("ui_month7"),
		["August"] = getDateText("ui_month8"),
		["September"] = getDateText("ui_month9"),
		["October"] = getDateText("ui_month11"),
		["November"] = getDateText("ui_month11"),
		["December"] = getDateText("ui_month12")
	}

	local function dateSubstr(text, indexOfFirstCharacter, indexOfLastCharacter, dot)
		local trimText = utf8Substr(text, indexOfFirstCharacter, indexOfLastCharacter)

		if dot and utf8Strlen(text) > indexOfLastCharacter then
			trimText = trimText .. "."
		end

		return trimText
	end

	self.dayNameShort = {}

	for dayName, text in pairs(self.dayName) do
		self.dayNameShort[dayName] = dateSubstr(text, 0, 3, false)
	end

	self.monthNameShort = {}

	for monthName, text in pairs(self.monthName) do
		self.monthNameShort[monthName] = dateSubstr(text, 0, 3, true)
	end

	local dateString = "%d/%m/%Y"

	if g_languageShort == "en" then
		dateString = "%Y-%m-%d"
	elseif g_languageShort == "de" then
		dateString = "%d.%m.%Y"
	elseif g_languageShort == "jp" then
		dateString = "%Y/%m/%d"
	end

	self.dateString = dateString

	self.bgOverlay = Overlay.new(g_baseUIFilename, 0, 0, 0, 0)
	self.bgOverlay:setUVs(g_colorBgUVs)

	self.bgOffset = 0.3
	self.textBold = false

	return self
end

function DateSetting:onLoad(filename)
	AdditionalSettingsUtil.prependedFunction(g_currentMission.hud.gameInfoDisplay, "drawTimeText", self, "drawTimeText")
end

function DateSetting:onStateChange(state, optionElement, loadFromSavegame)
	self:updateCurrentDateFormat(state, g_additionalSettingsManager:getSettingStateByName("hourFormat"))

	if optionElement ~= nil and optionElement.target ~= nil then
		local disabled = not g_currentMission.hud:getIsVisible() or state == 0

		optionElement.target.multiClockPosition:setDisabled(disabled)
		optionElement.target.checkClockBackground:setDisabled(disabled)
		optionElement.target.buttonDateColor:setDisabled(disabled)
	end
end

function DateSetting:onCreateElement(optionElement)
	local textHour = g_i18n:getText("additionalSettings_ui_currentTimeHour")
	local textDate = g_i18n:getText("additionalSettings_ui_currentTimeDate")
	local textDateShort = g_i18n:getText("additionalSettings_ui_currentTimeDateShort")
	local textDateLong = g_i18n:getText("additionalSettings_ui_currentTimeDateLong")

	optionElement:setTexts({
		g_i18n:getText("ui_off"),
		textHour,
		string.format("%s + %s", textHour, textDate),
		string.format("%s + %s (%s)", textHour, textDate, textDateShort),
		string.format("%s + %s (%s)", textHour, textDate, textDateLong)
	})
end

function DateSetting:onFrameOpen(optionElement)
	optionElement:setDisabled(not g_currentMission.hud:getIsVisible())
end

function DateSetting:updateCurrentDateFormat(state, hourFormat)
	local hour = "%H"
	local am_pm = ""

	if not hourFormat then
		hour = "%I"
		am_pm = " %p"
	end

	local dateFormat = hour .. ":%M:%S" .. am_pm

	if state == 2 then
		dateFormat = dateFormat .. " | " .. self.dateString
	elseif state == 3 or state == 4 then
		dateFormat = dateFormat .. " | dayName, monthDay monthName %Y"
	end

	self.dateFormat = dateFormat
end

function DateSetting.drawTimeText(setting, object)
	if setting.state ~= 0 then
		local dateFormat = setting.dateFormat

		if setting.state == 3 or setting.state == 4 then
			local monthDay = string.gsub(getDate("%d"), "^0+", "")
			local dayName = getDate("%A")
			local monthName = getDate("%B")
			local dayNameM = setting.dayName[dayName]
			local monthNameM = setting.monthName[monthName]

			if setting.state == 3 then
				dayNameM = setting.dayNameShort[dayName]
				monthNameM = setting.monthNameShort[monthName]
			end

			dateFormat = string.gsub(dateFormat, "monthDay", monthDay)
			dateFormat = string.gsub(dateFormat, "dayName", dayNameM or dayName)
			dateFormat = string.gsub(dateFormat, "monthName", monthNameM or monthName)
		end

		dateFormat = getDate(dateFormat)

		if setting.blinking then
			local seconds = tonumber(dateFormat:sub(7, 8))

			if seconds ~= nil then
				if seconds ~= setting.blinkSec then
					setting.blinkSec = seconds
					setting.blinkTime = 0
				else
					setting.blinkTime = setting.blinkTime + g_currentDt
				end

				if setting.blinkTime <= 250 or setting.blinkTime >= 750 then
					dateFormat = string.gsub(dateFormat, ":", " ")
				end
			end
		end

		local clockSettings = setting.clockSettings[g_additionalSettingsManager:getSettingStateByName("clockPosition")]
		local textSize = object.moneyTextSize * 0.8

		setTextBold(setting.textBold)
		setTextAlignment(clockSettings.horizontalAlignment)
		setTextVerticalAlignment(clockSettings.verticalAligment)

		local position = clockSettings.playerPosition

		if clockSettings.vehiclePosition ~= nil and g_currentMission.controlledVehicle ~= nil then
			position = clockSettings.vehiclePosition
		end

		local posX, posY = position.posX, position.posY

		local clockBackground = g_additionalSettingsManager:getSettingStateByName("clockBackground")

		if clockBackground then
			local bgOverlay = setting.bgOverlay
			local bgOffset = setting.bgOffset * textSize
			local width, height = getTextWidth(textSize, dateFormat), getTextHeight(textSize, dateFormat)
			local bgPosX, bgPosY = posX, posY - height * 0.12

			local horizontalAlignment = Overlay.ALIGN_HORIZONTAL_CENTER

			if clockSettings.horizontalAlignment == RenderText.ALIGN_RIGHT then
				horizontalAlignment = Overlay.ALIGN_HORIZONTAL_RIGHT
				posX = posX - bgOffset
			elseif clockSettings.horizontalAlignment == RenderText.ALIGN_LEFT then
				horizontalAlignment = Overlay.ALIGN_HORIZONTAL_LEFT
				posX = posX + bgOffset
			end

			local verticalAligment = Overlay.ALIGN_VERTICAL_MIDDLE

			if clockSettings.verticalAligment == RenderText.VERTICAL_ALIGN_TOP then
				verticalAligment = Overlay.ALIGN_VERTICAL_TOP
				posY = posY - bgOffset
			elseif clockSettings.verticalAligment == RenderText.VERTICAL_ALIGN_BOTTOM then
				verticalAligment = Overlay.ALIGN_VERTICAL_BOTTOM
				posY = posY + bgOffset
			end

			local backgroundColor = clockSettings.playerBgColor

			if clockSettings.vehicleBgColor ~= nil and g_currentMission.controlledVehicle ~= nil then
				backgroundColor = clockSettings.vehicleBgColor
			end

			bgOverlay:setPosition(bgPosX, bgPosY)
			bgOverlay:setDimension(width + bgOffset * 2, height + bgOffset * 2)
			bgOverlay:setColor(unpack(backgroundColor))
			bgOverlay:setAlignment(verticalAligment, horizontalAlignment)
			bgOverlay:draw()
		else
			local shOffset = textSize * HUDTextDisplay.SHADOW_OFFSET_FACTOR

			setTextColor(0, 0, 0, 1)
			renderText(posX + shOffset, posY - shOffset, textSize, dateFormat)
		end

		local clockColorSetting = g_additionalSettingsManager:getSettingByName("clockColor")

		setTextColor(unpack(clockColorSetting:getCurrentColor()))
		renderText(posX, posY, textSize, dateFormat)
		setTextAlignment(RenderText.ALIGN_LEFT)
		setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_BASELINE)
		setTextColor(1, 1, 1, 1)
	end
end


--
-- ClockPosition
--


ClockPositionSetting = {}

function ClockPositionSetting:new()
	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	return self
end

function ClockPositionSetting:onFrameOpen(optionElement)
	optionElement:setDisabled(not g_currentMission.hud:getIsVisible() or g_additionalSettingsManager:getSettingStateByName("date") == 0)
end


--
-- HourFormat
--


HourFormatSetting = {}

function HourFormatSetting:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.active = true
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	return self
end

function HourFormatSetting:onLoad(filename)
	AdditionalSettingsUtil.overwrittenFunction(g_currentMission.hud.gameInfoDisplay, "updateTime", self, "updateTime")
end

function HourFormatSetting:onFrameOpen(checkboxElement)
	checkboxElement:setDisabled(not g_currentMission.hud:getIsVisible())
end

function HourFormatSetting:onStateChange(state, checkboxElement, loadFromSavegame)
	local gameInfoDisplay = g_currentMission.hud.gameInfoDisplay
	local timeBoxWidth = GameInfoDisplay.SIZE.TIME_BOX[1]

	if not state then
		timeBoxWidth = GameInfoDisplay.SIZE.TIME_BOX[1] + 37.5
	end

	local boxWidth = gameInfoDisplay:scalePixelToScreenWidth(timeBoxWidth)
	local sepX, sepY = gameInfoDisplay.timeBox.separator:getPosition()

	gameInfoDisplay.timeBox.overlay.width = boxWidth

	gameInfoDisplay:storeScaledValues()
	gameInfoDisplay:updateSizeAndPositions()

	gameInfoDisplay.timeBox.separator:setPosition(sepX, sepY)

	local dateSettings = g_additionalSettingsManager:getSettingByName("date")

	dateSettings:updateCurrentDateFormat(dateSettings.state, state)
end

function HourFormatSetting.updateTime(setting, object, superFunc)
	local currentTime = object.environment.dayTime / 3600000
	local timeHours = math.floor(currentTime)
	local timeMinutes = math.floor((currentTime - timeHours) * 60)

	local timeHoursText = timeHours
	local am_pm = ""

	if not setting.active then
		am_pm = " AM"

		if timeHoursText == 0 then
			timeHoursText = 12
		elseif timeHoursText > 11 then
			if timeHoursText > 12 then
				timeHoursText = timeHoursText - 12
			end

			am_pm = " PM"
		end
	end

	object.timeText = string.format("%02d:%02d%s", timeHoursText, timeMinutes, am_pm)

	if object.missionInfo.timeScale < 1 then
		object.timeScaleText = string.format("%0.1f", object.missionInfo.timeScale)
	else
		object.timeScaleText = string.format("%d", object.missionInfo.timeScale)
	end

	object.monthText = g_i18n:formatDayInPeriod(nil, nil, true)

	object.seasonOverlay:setUVs(GuiUtils.getUVs(GameInfoDisplay.UV.SEASON[object.environment.currentSeason]))

	local hourRotation = -(currentTime % 12 / 12) * math.pi * 2
	local minutesRotation = -(currentTime - timeHours) * math.pi * 2

	object.clockHandSmall:setRotation(hourRotation)
	object.clockHandLarge:setRotation(minutesRotation)

	local isTimeScaleFast = object.missionInfo.timeScale > 1

	object.timeScaleArrow:setVisible(not isTimeScaleFast)
	object.timeScaleArrowFast:setVisible(isTimeScaleFast)
end


--
-- FadeEffect
--


FadeEffectSetting = {}

function FadeEffectSetting:new()
	AdditionalSettingsUtil.registerEventListener("onUpdate", self)
	AdditionalSettingsUtil.registerEventListener("onPreDraw", self)
	AdditionalSettingsUtil.overwrittenStaticFunction(g_env, "setCamera", self, "setCamera")
	AdditionalSettingsUtil.overwrittenFunction(g_gui, "changeScreen", self, "changeScreen")

	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	local fadeOverlay = Overlay.new(g_baseHUDFilename, 0, 0, 1, 1)

	fadeOverlay:setUVs(GuiUtils.getUVs(HUD.UV.AREA))
	fadeOverlay:setColor(0, 0, 0, 0)

	self.fadeScreenElement = HUDElement.new(fadeOverlay)
	self.fadeAnimation = TweenSequence.NO_SEQUENCE
	self.effectDuration = 0

	return self
end

function FadeEffectSetting:onUpdate(dt)
	if g_currentMission:getIsClient() then
		if not self.fadeAnimation:getFinished() then
			self.fadeAnimation:update(dt)
		end
	end
end

function FadeEffectSetting:onPreDraw()
	if self.fadeScreenElement:getVisible() then
		self.fadeScreenElement:draw()
	end
end

function FadeEffectSetting:onCreateElement(optionElement)
	local texts = {
		g_i18n:getText("ui_off")
	}

	for i = 100, 1000 + 0.0001, 100 do
		table.insert(texts, string.format("%d ms", i))
	end

	optionElement:setTexts(texts)
end

function FadeEffectSetting:getFadeEffectDurationFromIndex(index)
	return MathUtil.clamp(index * 100, 0, 1000)
end

function FadeEffectSetting:onStateChange(state, optionElement, loadFromSavegame)
	self.effectDuration = self:getFadeEffectDurationFromIndex(state)
end

function FadeEffectSetting:fadeScreen()
	if not g_currentMission.hud:getIsFading() and self.effectDuration ~= 0 then
		local seq = TweenSequence.new(self.fadeScreenElement)

		seq:addTween(Tween.new(self.fadeScreenElement.setAlpha, 1, 0, self.effectDuration))
		seq:start()

		self.fadeAnimation = seq
	end
end

function FadeEffectSetting.setCamera(setting, superFunc, camera)
	setting:fadeScreen()
	superFunc(camera)
end

function FadeEffectSetting.changeScreen(setting, object, superFunc, source, screenClass, returnScreenClass)
	local disableFadeEffect = object.currentGuiName == "ChatDialog"
	local retValue = superFunc(object, source, screenClass, returnScreenClass)

	if not disableFadeEffect and object.currentGuiName == "" then
		setting:fadeScreen()
	end

	return retValue
end


--
-- DialogBoxes
--


DialogBoxesSetting = {}

function DialogBoxesSetting:new()
	AdditionalSettingsUtil.registerEventListener("onUpdate", self)
	AdditionalSettingsUtil.prependedFunction(g_infoDialog, "setText", self, "setText")

	self.active = true
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	self.infoDialogs = {
		"shop_messageBoughtAnimals",
		"shop_messageBoughtChainsaw",
		"shop_messageBoughtPlaceable",
		"shop_messageConfigurationChanged",
		"shop_messageGardenCenterPurchaseReady",
		"shop_messageLeasingReady",
		"shop_messagePurchaseReady",
		"shop_messageReturnedVehicle",
		"shop_messageSoldAnimals",
		"shop_messageSoldObject",
		"shop_messageSoldVehicle",
		"shop_messageThanksForBuying",
		"ui_vehicleResetDone"
	}

	self.closeInfoDialog = false

	return self
end

function DialogBoxesSetting:onUpdate(dt)
	if g_currentMission:getIsClient() and not self.active and self.closeInfoDialog then
		g_infoDialog:close()

		if g_infoDialog.onOk ~= nil then
			if g_infoDialog.target ~= nil then
				g_infoDialog.onOk(g_infoDialog.target, g_infoDialog.args)
			else
				g_infoDialog.onOk(g_infoDialog.args)
			end
		end

		self.closeInfoDialog = false
	end
end

function DialogBoxesSetting.setText(setting, object, text)
	if not setting.active then
		for _, textToHide in pairs(setting.infoDialogs) do
			if text == g_i18n:getText(textToHide) then
				setting.closeInfoDialog = true
				break
			end
		end
	end
end


--
-- TrailerHUD
--


TrailerHUDSetting = {}

function TrailerHUDSetting:new()
	AdditionalSettingsUtil.registerEventListener("onPreDraw", self)
	AdditionalSettingsUtil.registerEventListener("onUpdate", self)
	AdditionalSettingsUtil.overwrittenStaticFunction(g_env, "setCamera", self, "setCamera")

	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	self.huds = {}
	self.hudEnableTime = 0

	self.time = {0.5, 1, 2, 3, 4, 5, 10}

	self.delay = 0

	return self
end

function TrailerHUDSetting.setCamera(setting, superFunc, camera)
	if setting.state ~= 0 then
		setting.delay = 250
	end

	superFunc(camera)
end

function TrailerHUDSetting:loadHUD()
	local hud = TrailerFillLevelHUD.new()

	if hud ~= nil then
		table.addElement(self.huds, hud)
		return hud
	end

	return nil
end

function TrailerHUDSetting:deleteHUD(hud)
	if hud ~= nil then
		table.removeElement(self.huds, hud)
		hud:delete()
	end
end

function TrailerHUDSetting:onUpdate(dt)
	if self.state ~= 0 then
		if self.delay > 0 then
			self.delay = self.delay - dt
		end

		for _, hud in pairs(self.huds) do
			if hud:getVisible() then
				hud:update(dt)
			end
		end
	end
end

function TrailerHUDSetting:onPreDraw()
	if self.state ~= 0 and self.delay <= 0 then
		for _, hud in pairs(self.huds) do
			if hud:getVisible() then
				hud:draw()
			end
		end
	end
end

function TrailerHUDSetting:onCreateElement(optionElement)
	local texts = {
		g_i18n:getText("ui_off")
	}

	for _, time in pairs(self.time) do
		table.insert(texts, string.format("%s s", tostring(time)))
	end

	optionElement:setTexts(texts)
end

function TrailerHUDSetting:onStateChange(state, optionElement, loadFromSavegame)
	local hudEnableTime = 0

	if state ~= 0 then
		hudEnableTime = self.time[state] * 1000
	end

	self.hudEnableTime = hudEnableTime
end


--
-- VehicleCamera
--


VehicleCameraSetting = {}

function VehicleCameraSetting:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	self.yaw = 0
	self.pitch = 0
	self.value = 0

	return self
end

function VehicleCameraSetting:onLoad(filename)
	AdditionalSettingsUtil.overwrittenFunction(VehicleCamera, "actionEventLookLeftRight", self, "actionEventLookLeftRight")
	AdditionalSettingsUtil.overwrittenFunction(VehicleCamera, "actionEventLookUpDown", self, "actionEventLookUpDown")
end

function VehicleCameraSetting:onCreateElement(optionElement)
	local texts = {
		g_i18n:getText("ui_off")
	}

	for i = 10, 100 + 0.0001, 10 do
		table.insert(texts, string.format("%d%%", i))
	end

	optionElement:setTexts(texts)
end

function VehicleCameraSetting:getSmoothCameraValueFromIndex(index)
	return MathUtil.clamp(5 - index * 0.35, 1.5, 5)
end

function VehicleCameraSetting:onStateChange(state, optionElement, loadFromSavegame)
	self.value = self:getSmoothCameraValueFromIndex(state)
end

function VehicleCameraSetting:getSmoothValue(pitch, inputValue, isMouse)
	if isMouse then
		inputValue = inputValue * 0.001 * 16.666
	else
		inputValue = inputValue * 0.001 * g_currentDt
	end

	self[pitch] = inputValue + math.pow(0.99579, g_currentDt * self.value) * (self[pitch] - inputValue)

	return self[pitch]
end

function VehicleCameraSetting.actionEventLookLeftRight(setting, object, superFunc, actionName, inputValue, callbackState, isAnalog, isMouse)
	if setting.state == 0 then
		superFunc(object, actionName, inputValue, callbackState, isAnalog, isMouse)
	else
		object.lastInputValues.leftRight = object.lastInputValues.leftRight + setting:getSmoothValue("yaw", inputValue, isMouse)
	end
end

function VehicleCameraSetting.actionEventLookUpDown(setting, object, superFunc, actionName, inputValue, callbackState, isAnalog, isMouse)
	if setting.state == 0 then
		superFunc(object, actionName, inputValue, callbackState, isAnalog, isMouse)
	else
		object.lastInputValues.upDown = object.lastInputValues.upDown + setting:getSmoothValue("pitch", inputValue, isMouse)
	end
end


--
-- PlayerCamera
--


PlayerCameraSetting = {}

function PlayerCameraSetting:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	self.yaw = 0
	self.pitch = 0
	self.value = 0

	return self
end

function PlayerCameraSetting:onLoad(filename)
	AdditionalSettingsUtil.overwrittenFunction(Player, "onInputLookLeftRight", self, "onInputLookLeftRight")
	AdditionalSettingsUtil.overwrittenFunction(Player, "onInputLookUpDown", self, "onInputLookUpDown")
end

function PlayerCameraSetting:onCreateElement(optionElement)
	local texts = {
		g_i18n:getText("ui_off")
	}

	for i = 10, 100 + 0.0001, 10 do
		table.insert(texts, string.format("%d%%", i))
	end

	optionElement:setTexts(texts)
end

function PlayerCameraSetting:getSmoothCameraValueFromIndex(index)
	return MathUtil.clamp(5 - index * 0.35, 1.5, 5)
end

function PlayerCameraSetting:onStateChange(state, optionElement, loadFromSavegame)
	self.value = self:getSmoothCameraValueFromIndex(state)
end

function PlayerCameraSetting:getSmoothValue(pitch, inputValue, isMouse)
	if isMouse then
		inputValue = inputValue * 0.001 * 16.666
	else
		inputValue = inputValue * 0.001 * g_currentDt
	end

	self[pitch] = inputValue + math.pow(0.99579, g_currentDt * self.value) * (self[pitch] - inputValue)

	return self[pitch]
end

function PlayerCameraSetting.onInputLookLeftRight(setting, object, superFunc, actionName, inputValue, callbackState, isAnalog, isMouse)
	if setting.state == 0 then
		superFunc(object, actionName, inputValue, callbackState, isAnalog, isMouse)
	else
		if not object.lockedInput then
			object.inputInformation.yawCamera = object.inputInformation.yawCamera + setting:getSmoothValue("yaw", inputValue, isMouse)
		end

		object.inputInformation.isMouseRotation = isMouse
	end
end

function PlayerCameraSetting.onInputLookUpDown(setting, object, superFunc, actionName, inputValue, callbackState, isAnalog, isMouse)
	if setting.state == 0 then
		superFunc(object, actionName, inputValue, callbackState, isAnalog, isMouse)
	else
		if not object.lockedInput then
			local pitchValue = g_gameSettings:getValue("invertYLook") and -inputValue or inputValue

			object.inputInformation.pitchCamera = object.inputInformation.pitchCamera + setting:getSmoothValue("pitch", pitchValue, isMouse)
		end
	end
end


--
-- EasyMotorStart
--


EasyMotorStartSetting = {}

function EasyMotorStartSetting:new()
	self.active = false
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	return self
end

function EasyMotorStartSetting:onFrameOpen(checkboxElement)
	checkboxElement:setDisabled(g_currentMission.missionInfo.automaticMotorStartEnabled)
end


--
-- Autostart
--

AutostartSetting = {}

function AutostartSetting:new()
	AdditionalSettingsUtil.appendedFunction(g_mpLoadingScreen, "onReadyToStart", self, "onReadyToStart")

	self.active = false
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	return self
end

function AutostartSetting:onCreateElement(checkboxElement)
	if StartParams.getIsSet("autoStart") then
		checkboxElement:setDisabled(true)

		local tooltipTextElement = checkboxElement.elements[6]

		tooltipTextElement:setText(string.format("%s\n*%s", tooltipTextElement:getText(), g_i18n:getText("additionalSettings_warning_autostart")))
	end
end

function AutostartSetting.onReadyToStart(setting, object)
	if setting.active and g_currentMission:canStartMission() and g_dedicatedServer == nil and not g_gui:getIsDialogVisible() then
		object:onClickOk()
	end
end


--
-- StoreItems
--


StoreItemsSetting = {}

function StoreItemsSetting:new()
	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	return self
end

function StoreItemsSetting:getIsItemVisible(item)
	local state = self.state

	if state == 0 or item == nil then
		return false
	end

	local isMod = false
	local isDlc = false

	if item.customEnvironment ~= nil then
		local isDLCEnv = string.startsWith(item.customEnvironment, g_uniqueDlcNamePrefix)

		isMod = not isDLCEnv
		isDlc = isDLCEnv
	end

	if state == 1 then
		return not isMod and not isDlc
	elseif state == 2 then
		return isMod or isDlc
	elseif state == 3 then
		return isMod
	else
		return isDlc
	end
end

function StoreItemsSetting:onStateChange(state, optionElement, loadFromSavegame)
	local itemsCategoryName = ""

	if state ~= 0 and optionElement ~= nil then
		itemsCategoryName = string.format(" (%s)", optionElement.texts[state + 1])
	end

	local shopMenu = g_currentMission.shopMenu
	local clickItemCategoryCallback = shopMenu:makeSelfCallback(self.onClickItemCategory)
	local selectCategoryCallback = shopMenu:makeSelfCallback(shopMenu.onSelectCategory)
	local shopPages = {
		[shopMenu.pageShopBrands] = {
			shopMenu.shopController:getBrands(),
			shopMenu:makeSelfCallback(self.onClickBrand),
			selectCategoryCallback,
			GuiUtils.getUVs(ShopMenu.TAB_UV.BRANDS),
			shopMenu.l10n:getText(ShopMenu.L10N_SYMBOL.HEADER_BRANDS) .. itemsCategoryName,
			ShopMenu.BRAND_IMAGE_HEIGHT_WIDTH_RATIO
		},
		[shopMenu.pageShopVehicles] = {
			shopMenu.shopController:getVehicleCategories(),
			clickItemCategoryCallback,
			selectCategoryCallback,
			GuiUtils.getUVs(ShopMenu.TAB_UV.VEHICLES),
			shopMenu.l10n:getText(ShopMenu.L10N_SYMBOL.HEADER_VEHICLES) .. itemsCategoryName
		},
		[shopMenu.pageShopTools] = {
			shopMenu.shopController:getToolCategories(),
			clickItemCategoryCallback,
			selectCategoryCallback,
			GuiUtils.getUVs(ShopMenu.TAB_UV.TOOLS),
			shopMenu.l10n:getText(ShopMenu.L10N_SYMBOL.HEADER_TOOLS) .. itemsCategoryName
		},
		[shopMenu.pageShopObjects] = {
			shopMenu.shopController:getObjectCategories(),
			clickItemCategoryCallback,
			selectCategoryCallback,
			GuiUtils.getUVs(ShopMenu.TAB_UV.OBJECTS),
			shopMenu.l10n:getText(ShopMenu.L10N_SYMBOL.HEADER_OBJECTS) .. itemsCategoryName
		}
	}

	for page, attr in pairs(shopPages) do
		if state ~= 0 then
			local visibleCategories = {}

			for _, category in ipairs(attr[1]) do
				local items = {}

				if page == shopMenu.pageShopBrands then
					items = shopMenu.shopController:getItemsByBrand(category.id)
				else
					items = shopMenu.shopController:getItemsByCategory(category.id)
				end

				local addCategory = false

				for i = 1, #items do
					if not addCategory and self:getIsItemVisible(items[i].storeItem) then
						table.insert(visibleCategories, category)
						addCategory = true
					end
				end
			end

			attr[1] = visibleCategories
		end

		page:reset()
		page:initialize(unpack(attr))
	end
end

function StoreItemsSetting.onClickBrand(object, brandId, brandCategoryIconUVs, brandCategoryDisplayName, categoryDisplayName)
	object.isShowingOwnedPage = false
	object.isShowingLeasedPage = false

	local brandItems = object.shopController:getItemsByBrand(brandId)
	local currentDisplayItems = brandItems

	local setting = g_additionalSettingsManager:getSettingByName("storeItems")

	if setting.state ~= 0 then
		local displayItems = {}

		for i = 1, #brandItems do
			if setting:getIsItemVisible(brandItems[i].storeItem) then
				table.insert(displayItems, brandItems[i])
			end
		end

		currentDisplayItems = displayItems
	end

	object.currentDisplayItems = currentDisplayItems
	object.pageShopItemDetails:setDisplayItems(currentDisplayItems, false)
	object.pageShopItemDetails:setCategory(brandCategoryIconUVs, brandCategoryDisplayName, categoryDisplayName)
	object:setDetailButtons()
	object:pushDetail(object.pageShopItemDetails)
end

function StoreItemsSetting.onClickItemCategory(object, categoryName, baseCategoryIconUVs, baseCategoryDisplayName, categoryDisplayName, filter)
	object.isShowingOwnedPage = false
	object.isShowingLeasedPage = false

	local categoryItems = object.shopController:getItemsByCategory(categoryName)
	local currentDisplayItems = categoryItems

	local setting = g_additionalSettingsManager:getSettingByName("storeItems")

	if setting.state ~= 0 then
		local displayItems = {}

		for i = 1, #categoryItems do
			if setting:getIsItemVisible(categoryItems[i].storeItem) then
				table.insert(displayItems, categoryItems[i])
			end
		end

		currentDisplayItems = displayItems
	end

	object.currentCategoryName = categoryName
	object.currentDisplayItems = currentDisplayItems
	object.pageShopItemDetails:setDisplayItems(object.currentDisplayItems, false)
	object.pageShopItemDetails:setCategory(baseCategoryIconUVs, baseCategoryDisplayName, categoryDisplayName)
	object:setDetailButtons()
	object:pushDetail(object.pageShopItemDetails)
end


--
-- Lighting
--


LightingSetting = {}

function LightingSetting:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)
	AdditionalSettingsUtil.registerEventListener("onDelete", self)

	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	return self
end

function LightingSetting:onLoad(filename)
	AdditionalSettingsUtil.overwrittenFunction(g_currentMission.environment, "setCustomLighting", self, "setCustomLighting")
	AdditionalSettingsUtil.overwrittenStaticFunction(g_env, "setEnvMap", self, "setEnvMap")

	local lightingDirectory = g_additionalSettingsManager.modSettings.directory .. "lighting/"

	createFolder(lightingDirectory)

	local defaultLightingDirectory = lightingDirectory .. "default/"

	createFolder(defaultLightingDirectory)

	AdditionalSettingsUtil.copyFiles(g_additionalSettingsManager.baseDirectory .. "lighting/", defaultLightingDirectory, {"colorGrading.xml", "colorGradingNight.xml", "lighting.xml", "name.xml"}, false)

	self.customLighting, self.customLightingTexts = self:loadCustomLightingConfigurations(lightingDirectory)
	self.lightingDirectory = lightingDirectory

	addConsoleCommand("asReloadCustomLighting", "", "consoleCommandReloadCustomLighting", self)
end

function LightingSetting:onDelete()
	removeConsoleCommand("asReloadCustomLighting")
end

function LightingSetting:onStateChange(state, optionElement, loadFromSavegame)
	local lighting
	local customLighting = self:getCurrentLighting()

	if customLighting ~= nil then
		lighting = customLighting.lighting
	end

	g_currentMission.environment:setCustomLighting(lighting)
end

function LightingSetting:onCreateElement(optionElement)
	optionElement:setTexts(self.customLightingTexts)

	local toolTipelement = optionElement.elements[6]
	toolTipelement:setText(string.format(toolTipelement:getText(), self.lightingDirectory))
end

function LightingSetting:onSaveSetting(xmlFile, key)
	local id = g_currentMission.missionInfo.map.id
	local isMap = true

	local customLighting = self:getCurrentLighting()

	if customLighting ~= nil then
		id = customLighting.id
		isMap = customLighting.isMap

		xmlFile:setString(key .. "#id", id)
		xmlFile:setBool(key .. "#isMap", isMap)
	end

	return true
end

function LightingSetting:onLoadSetting(xmlFile, key)
	local state = 0

	local id = xmlFile:getString(key .. "#id")
	local isMap = xmlFile:getBool(key .. "#isMap")

	if id ~= nil and isMap ~= nil then
		for i, customLighting in pairs(self.customLighting) do
			if customLighting.id == id and customLighting.isMap == isMap then
				state = i
			end
		end
	end

	return true, state
end

function LightingSetting:loadCustomLightingConfigurations(directory)
	local customLighting = {}
	local customLightingTexts = {g_i18n:getText("ui_off")}
	local baseDirectoryBackup = g_currentMission.baseDirectory
	local files = Files.new(directory)

	setFileLogPrefixTimestamp(false)

	for _, v in pairs(files.files) do
		local filename = v.filename

		if not v.isDirectory then
			filename = v.filename:sub(1, -5)
		end

		local baseDirectory = directory .. filename .. "/"
		local xmlFilename = baseDirectory .. "lighting.xml"
		local baseKey = "lighting"

		if not fileExists(xmlFilename) then
			xmlFilename = baseDirectory .. "environment.xml"
			baseKey = "environment.lighting"
		end

		local lightingXML = XMLFile.loadIfExists("lightingXML", xmlFilename)

		if lightingXML ~= nil then
			g_currentMission.baseDirectory = baseDirectory

			local lighting = Lighting.new(g_currentMission.environment)

			lighting:load(lightingXML, baseKey)

			local orgEnvMapTimes = #lighting.envMapTimes
			local orgEnvMapBasePath = lighting.envMapBasePath

			if lightingXML:getBool(baseKey .. ".envMap#attr10", #lighting.envMapTimes == 0 or lighting.envMapBasePath == nil) then
				self:setLightingAttributes(lighting)
			end

			local text = filename
			local nameXML = XMLFile.loadIfExists("nameXML", baseDirectory .. "name.xml")

			if nameXML ~= nil then
				text = nameXML:getI18NValue("name", "", g_additionalSettingsManager.customEnvironment, true)
				nameXML:delete()
			end

			table.insert(customLighting, {lighting = lighting, id = filename, isMap = false})
			table.insert(customLightingTexts, string.format("%s (%s)", text, g_i18n:getText("additionalSettings_ui_folder")))

			AdditionalSettingsUtil.info("Lighting configuration loaded (%s):\n  -lighting: '%s'\n  -colorGradingDay: '%s'\n  -colorGradingNight: '%s'\n  -envMapBasePath: '%s'\n  -envMapTimes: '%d'\n  -useStoreEnvMap: '%s'\n", text, xmlFilename, lighting.colorGradingDay, lighting.colorGradingNight, orgEnvMapBasePath, orgEnvMapTimes, lighting.attr10 == true)

			lightingXML:delete()
		else
			AdditionalSettingsUtil.error("File '%s' not found!", xmlFilename)
		end
	end

	for i = 1, g_mapManager:getNumOfMaps() do
		local map = g_mapManager:getMapDataByIndex(i)
		local mapXMLFilename = Utils.getFilename(map.mapXMLFilename, map.baseDirectory)
		local mapXML = XMLFile.load("MapXML", mapXMLFilename)

		if mapXML ~= nil then
			local environmentXMLFilename = Utils.getFilename(mapXML:getString("map.environment#filename"), map.baseDirectory)

			if environmentXMLFilename == "data/maps/mapUS/environment.xml" and map.title ~= g_i18n:getText("mapUS_title") or environmentXMLFilename == "data/maps/mapFR/environment.xml" and map.title ~= g_i18n:getText("mapFR_title") or environmentXMLFilename == "data/maps/mapAlpine/environment.xml" and map.title ~= g_i18n:getText("mapDE_title") then
				AdditionalSettingsUtil.info("Default lighting configuration: (%s) '%s'.\n", map.title, environmentXMLFilename)
			else
				local environmentXML = XMLFile.load("environmentXML", environmentXMLFilename)

				if environmentXML ~= nil then
					g_currentMission.baseDirectory = map.baseDirectory

					local lighting = nil

					if map == g_currentMission.missionInfo.map then
						lighting = g_currentMission.environment.baseLighting
					else
						lighting = Lighting.new(g_currentMission.environment)
						lighting:load(environmentXML, "environment.lighting")
					end

					local orgEnvMapTimes = #lighting.envMapTimes
					local orgEnvMapBasePath = lighting.envMapBasePath

					if environmentXML:getBool("environment.lighting.envMap#attr10", #lighting.envMapTimes == 0 or lighting.envMapBasePath == nil) then
						self:setLightingAttributes(lighting)
					end

					table.insert(customLighting, {lighting = lighting, id = map.id, isMap = true})
					table.insert(customLightingTexts, string.format("%s (%s)", map.title, g_i18n:getText("ui_map")))

					AdditionalSettingsUtil.info("Lighting configuration loaded (%s):\n  -lighting: '%s'\n  -colorGradingDay: '%s'\n  -colorGradingNight: '%s'\n  -envMapBasePath: '%s'\n  -envMapTimes: '%d'\n  -useStoreEnvMap: '%s'\n", map.title, environmentXMLFilename, lighting.colorGradingDay, lighting.colorGradingNight, orgEnvMapBasePath, orgEnvMapTimes, lighting.attr10 == true)

					environmentXML:delete()
				end
			end
		end

		mapXML:delete()
	end

	setFileLogPrefixTimestamp(true)

	g_currentMission.baseDirectory = baseDirectoryBackup

	local beaseLighting = g_currentMission.environment.baseLighting

	if #beaseLighting.envMapTimes == 0 or beaseLighting.envMapBasePath == nil then
		self:setLightingAttributes(beaseLighting)
	end

	return customLighting, customLightingTexts
end

function LightingSetting:setLightingAttributes(lighting)
	lighting.attr10 = true
	lighting.envMapBasePath = "data/store/ui/envMaps/shop/"
	lighting.envMapTimes = {0}
end

function LightingSetting:getCurrentLighting()
	if self.state ~= 0 then
		return self.customLighting[self.state]
	end

	return nil
end

function LightingSetting.setCustomLighting(setting, object, superFunc, lighting)
	if lighting == nil then
		local customLighting = setting:getCurrentLighting()

		if customLighting ~= nil then
			lighting = customLighting.lighting
		end
	end

	superFunc(object, lighting)
end

function LightingSetting.setEnvMap(setting, superFunc, envMapTime0Cloud0, envMapTime0Cloud1, envMapTime1Cloud0, envMapTime1Cloud1, blendWeight0, blendWeight1, blendWeight2, blendWeight3, force, attr10)
	if attr10 == false then
		local customLighting = setting:getCurrentLighting()

		if customLighting ~= nil then
			if customLighting.lighting.attr10 then
				attr10 = true
			end
		elseif g_currentMission.environment.lighting == g_currentMission.environment.baseLighting then
			if g_currentMission.environment.baseLighting.attr10 then
				attr10 = true
			end
		end
	end

	superFunc(envMapTime0Cloud0, envMapTime0Cloud1, envMapTime1Cloud0, envMapTime1Cloud1, blendWeight0, blendWeight1, blendWeight2, blendWeight3, force, attr10)
end

function LightingSetting:consoleCommandReloadCustomLighting()
	self.state = 0
	self:onStateChange(0, nil, false)

	g_currentMission.xmlFile = loadXMLFileFromMemory("mapXML", "<map/>")

	self.customLighting, self.customLightingTexts = self:loadCustomLightingConfigurations(self.lightingDirectory)

	delete(g_currentMission.xmlFile)
	g_currentMission.xmlFile = nil

	local optionElement = g_additionalSettingsManager.getUIElement(self)

	optionElement:setTexts(self.customLightingTexts)
	optionElement:setState(1)

	return string.format("Custom lighting settings updated, available configurations: %d", #self.customLighting)
end


--
-- DoF
--


DoFSetting = {}

function DoFSetting:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)
	AdditionalSettingsUtil.registerEventListener("onMissionStarted", self)
	AdditionalSettingsUtil.registerEventListener("onUpdate", self)

	self.active = true
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	self.updateDoF = false

	return self
end

function DoFSetting:onLoad(filename)
	self.active = g_currentMission.environment.depthOfField
end

function DoFSetting:onMissionStarted()
	self.defaultStateBackup = g_depthOfFieldManager.defaultState
end

function DoFSetting:onUpdate(dt)
	if self.updateDoF and g_depthOfFieldManager:getIsDoFChangeAllowed() then
		g_depthOfFieldManager:reset()
		self.updateDoF = false
	end
end

function DoFSetting:onStateChange(state, checkboxElement, loadFromSavegame)
	local initialState = {0, 0, 0, math.huge, math.huge, false}

	if state then
		initialState = self.defaultStateBackup
	end

	g_depthOfFieldManager.initialState = initialState
	self.updateDoF = true
end


--
-- WildAnimals
--


WildAnimalslSetting = {}

function WildAnimalslSetting:new()
	self.active = true
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	return self
end

function WildAnimalslSetting:onCreateElement(checkboxElement)
	checkboxElement:setDisabled(g_currentMission.wildlifeSpawner == nil)
end

function WildAnimalslSetting:onStateChange(state, checkboxElement, loadFromSavegame)
	local wildlifeSpawner = g_currentMission.wildlifeSpawner

	if wildlifeSpawner ~= nil then
		wildlifeSpawner.isEnabled = state

		if not state then
			wildlifeSpawner:removeAllAnimals()
		end
	end
end

function WildAnimalslSetting:onFrameOpen(checkboxElement)
	local wildlifeSpawner = g_currentMission.wildlifeSpawner

	if wildlifeSpawner ~= nil then
		self.active = wildlifeSpawner.isEnabled
	end
end


--
-- CameraCollisions
--


CameraCollisionsSetting = {}

function CameraCollisionsSetting:new()
	AdditionalSettingsUtil.overwrittenFunction(VehicleCamera, "getCollisionDistance", self, "getCollisionDistance", not g_modIsLoaded.FS22_disableVehicleCameraCollision)

	self.active = true
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	return self
end

function CameraCollisionsSetting:onCreateElement(checkboxElement)
	checkboxElement:setVisible(not g_modIsLoaded.FS22_disableVehicleCameraCollision)
end

function CameraCollisionsSetting.getCollisionDistance(setting, object, superFunc)
	if not setting.active then
		return false, nil, nil, nil, nil, nil
	end

	return superFunc(object)
end


--
-- GuiCamera
--


GuiCameraSetting = {}

function GuiCameraSetting:new()
	AdditionalSettingsUtil.overwrittenFunction(GuiTopDownCamera, "getMouseEdgeScrollingMovement", self, "getMouseEdgeScrollingMovement")

	self.active = true
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	return self
end

function GuiCameraSetting.getMouseEdgeScrollingMovement(setting, object, superFunc)
	if not setting.active then
		return 0, 0
	end

	return superFunc(object)
end


--
-- QuitGame
--


QuitGameSetting = {}

function QuitGameSetting:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)

	self.loadingState = AdditionalSettingsManager.LOADING_STATE.NO_LOAD

	return self
end

function QuitGameSetting:onLoad(filename)
	local inGameMenu = g_currentMission.inGameMenu
	local pageSettingsGame = inGameMenu.pageSettingsGame

	local function onButtonQuitGame()
		if inGameMenu.isSaving then
			return
		end

		if not inGameMenu.playerAlreadySaved and not (inGameMenu.missionDynamicInfo.isMultiplayer and inGameMenu.missionDynamicInfo.isClient) then
			g_gui:showYesNoDialog({
				text = g_i18n:getText(InGameMenu.L10N_SYMBOL.END_WITHOUT_SAVING),
				callback = function (yes)
					if yes then
						if inGameMenu.missionDynamicInfo.isMultiplayer and inGameMenu.isServer then
							inGameMenu.server:broadcastEvent(ShutdownEvent.new())
						end
						doExit()
					end
				end
			})
		else
			doExit()
		end
	end

	inGameMenu.quitGameButton = {
		showWhenPaused = true,
		inputAction = InputAction.MENU_EXTRA_2,
		text = g_i18n:getText("additionalSettings_button_quitGame"),
		callback = onButtonQuitGame
	}

	table.insert(inGameMenu.defaultMenuButtonInfo, inGameMenu.quitGameButton)
	inGameMenu.defaultMenuButtonInfoByActions[InputAction.MENU_EXTRA_2] = inGameMenu.quitGameButton
	inGameMenu.defaultButtonActionCallbacks[InputAction.MENU_EXTRA_2] = onButtonQuitGame

	pageSettingsGame.updateButtons = Utils.overwrittenFunction(pageSettingsGame.updateButtons, function (object, superFunc)
		object.menuButtonInfo = {
			object.backButtonInfo,
			object.saveButton,
			object.quitButton,
			inGameMenu.quitGameButton
		}

		if object.hasMasterRights and g_currentMission.missionDynamicInfo.isMultiplayer then
			table.insert(object.menuButtonInfo, object.serverSettingsButton)
		end

		object:setMenuButtonInfoDirty()
	end)

	pageSettingsGame:updateButtons()
end


--
-- ClockColorSetting
--


ClockColorSetting = {}

function ClockColorSetting:new()
	AdditionalSettingsUtil.registerEventListener("onLoad", self)
	AdditionalSettingsUtil.registerEventListener("onDelete", self)

	self.state = 0
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START

	return self
end

function ClockColorSetting:onLoad(filename)
	self.clockFilename = g_additionalSettingsManager.modSettings.directory .. "clockColors.xml"

	local forceCopy = fileExists(self.clockFilename) and getFileMD5(self.clockFilename, "clockColors.xml") == "76b6d4e561abb5f4d7841dfef20dc7b2"

	AdditionalSettingsUtil.copyFiles(g_additionalSettingsManager.baseDirectory, g_additionalSettingsManager.modSettings.directory, {"clockColors.xml"}, forceCopy)

	self:loadColors()
	addConsoleCommand("asReloadClockColors", "", "consoleCommandReloadClockColors", self)
end

function ClockColorSetting:onDelete()
	removeConsoleCommand("asReloadClockColors")
end

function ClockColorSetting:loadColors()
	self.colors = {}

	local useDefaultColors = false
	local colorXML = XMLFile.loadIfExists("colorXML", self.clockFilename)

	if colorXML ~= nil then
		useDefaultColors = colorXML:getBool("colors#useDefaultColors", false)

		if colorXML:hasProperty("colors.color(0)") then
			colorXML:iterate("colors.color", function (index, key)
				local c = ConfigurationUtil.getColorFromString(colorXML:getString(key .. "#color"))
				local n =  g_i18n:convertText(colorXML:getString(key .. "#name", ""), g_additionalSettingsManager.customEnvironment)

				c[4] = 1
				table.insert(self.colors, {color = c, name = n})
			end)
		end

		colorXML:delete()
	else
		AdditionalSettingsUtil.error("File '%s' not found!", self.clockFilename)
	end

	if #self.colors == 0 then
		useDefaultColors = true
	end

	if useDefaultColors then
		for _, color in pairs(g_vehicleColors) do
			local c = g_brandColorManager:getBrandColorByName(color.brandColor)
			local n = g_i18n:convertText(color.name)

			table.insert(self.colors, {color = c, name = n})
		end
	end
end

function ClockColorSetting:onFrameOpen(buttonElement)
	buttonElement:setDisabled(not g_currentMission.hud:getIsVisible() or g_additionalSettingsManager:getSettingStateByName("date") == 0)
end

function ClockColorSetting:onStateChange(state, buttonElement, loadFromSavegame)
	if loadFromSavegame then
		if state > #self.colors then
			self.state = 0
		end

		self:updateColorButton()
	end
end

function ClockColorSetting:onCreateElement(buttonElement)
	self.changeColorButton = buttonElement.elements[2]
end

function ClockColorSetting:onClickButton(buttonElement)
	g_gui:showColorPickerDialog({
		colors = self.colors,
		defaultColor = self.colors[self.state + 1].color,
		callback = self.onPickedColor,
		target = self
	})

	return true
end

function ClockColorSetting:onPickedColor(colorIndex, args)
	if colorIndex ~= nil then
		if colorIndex > #self.colors then
			colorIndex = 1
		end

		self.state = colorIndex - 1

		self:updateColorButton()
		g_currentMission.guiSoundPlayer:playSample(GuiSoundPlayer.SOUND_SAMPLES.CONFIG_SPRAY)
	end
end

function ClockColorSetting:getCurrentColor()
	local color = self.colors[self.state + 1]

	if color ~= nil then
		return color.color
	end

	return self.colors[1].color
end

function ClockColorSetting:updateColorButton()
	if self.changeColorButton ~= nil then
		self.changeColorButton:setImageColor(nil, unpack(self:getCurrentColor()))
	end
end

function ClockColorSetting:consoleCommandReloadClockColors()
	self.state = 0

	self:updateColorButton()
	self:loadColors()

	return "Done!"
end


--
-- FramerateLimiterSetting
--


FramerateLimiterSetting = {}

function FramerateLimiterSetting:new()
	self.state = 4

	local loadingState = AdditionalSettingsManager.LOADING_STATE.NO_LOAD

	if g_dedicatedServer == nil and (g_isDevelopmentVersion or g_maxModDescVersion <= 65) then
		loadingState = AdditionalSettingsManager.LOADING_STATE.MISSION_START
		Platform.hasAdjustableFrameLimit = false
	end

	self.loadingState = loadingState
	self.limit = {30, 40, 50, 60, 75, 100, 120, 144}

	return self
end

function FramerateLimiterSetting:onCreateElement(optionElement)
	optionElement:setVisible(self.loadingState ~= AdditionalSettingsManager.LOADING_STATE.NO_LOAD)

	local texts = {
		g_i18n:getText("ui_off")
	}

	for _, limit in pairs(self.limit) do
		table.insert(texts, tostring(limit))
	end

	optionElement:setTexts(texts)
end

function FramerateLimiterSetting:onStateChange(state, optionElement, loadFromSavegame)
	local maxFrameLimit = DedicatedServer.MAX_FRAME_LIMIT

	DedicatedServer.MAX_FRAME_LIMIT = self.limit[state] or math.huge
	DedicatedServer:raiseFramerate()
	DedicatedServer.MAX_FRAME_LIMIT = maxFrameLimit
end


--
-- ClockBackgroundSetting
--


ClockBackgroundSetting = {}

function ClockBackgroundSetting:new()
	self.active = true
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	return self
end

function ClockBackgroundSetting:onFrameOpen(checkboxElement)
	checkboxElement:setDisabled(not g_currentMission.hud:getIsVisible() or g_additionalSettingsManager:getSettingStateByName("date") == 0)
end


--
-- BlinkingWarningsSetting
--


BlinkingWarningsSetting = {}

function BlinkingWarningsSetting:new()
	self.active = true
	self.loadingState = AdditionalSettingsManager.LOADING_STATE.LOAD_MAP

	AdditionalSettingsUtil.overwrittenFunction(g_currentMission, "showBlinkingWarning", self, "showBlinkingWarning")

	return self
end

function BlinkingWarningsSetting.showBlinkingWarning(setting, object, superFunc, text, duration, priority)
	if setting.active then
		superFunc(object, text, duration, priority)
	end
end