--
-- AdditionalSettingsManager
--
-- @author Rockstar
-- @date 27/03/2021
--

--
-- @fs22 24/11/2021
--


AdditionalSettingsManager = {
	LOADING_STATE = {
		NO_LOAD = 0,
		LOAD_MAP = 1,
		LOAD_MAP_FINISHED = 2,
		MISSION_START = 3
	}
}

local AdditionalSettingsManager_mt = Class(AdditionalSettingsManager)

function AdditionalSettingsManager.new(modDir, modSettingsDir, modName, customMt)
	local self = setmetatable({}, customMt or AdditionalSettingsManager_mt)

	AdditionalSettingsUtil.registerEvent("onLoad")
	AdditionalSettingsUtil.registerEvent("onLoadFinished")
	AdditionalSettingsUtil.registerEvent("onMissionStarted")
	AdditionalSettingsUtil.registerEvent("onDelete")
	AdditionalSettingsUtil.registerEvent("onUpdate")
	AdditionalSettingsUtil.registerEvent("onDraw")
	AdditionalSettingsUtil.registerEvent("onPreDraw")

	addModEventListener(self)

	g_currentMission:registerToLoadOnMapFinished(self)
	g_currentMission:registerObjectToCallOnMissionStart(self)

	createFolder(modSettingsDir)

	self.modSettings = {
		directory = modSettingsDir,
		xmlFilename = modSettingsDir .. "additionalSettings.xml"
	}

	self.baseDirectory = modDir
	self.customEnvironment = modName

	self.modVersion = g_modManager:getModByName(modName).version
	self.resetAllowed = false

	self.addittionalSettings = self.loadAdditionalSettings()
	self.addModEnvironmentTexts()
	self.addSettingsPage(modDir)

	return self
end

function AdditionalSettingsManager.loadAdditionalSettings()
	return {
		hud = HUDSetting:new(),
		crosshair = CrosshairSetting:new(),
		date = DateSetting:new(),
		clockPosition = ClockPositionSetting:new(),
		hourFormat = HourFormatSetting:new(),
		fadeEffect = FadeEffectSetting:new(),
		dialogBoxes = DialogBoxesSetting:new(),
		trailerHUD = TrailerHUDSetting:new(),
		vehicleCamera = VehicleCameraSetting:new(),
		playerCamera = PlayerCameraSetting:new(),
		easyMotorStart = EasyMotorStartSetting:new(),
		autostart = AutostartSetting:new(),
		storeItems = StoreItemsSetting:new(),
		lighting = LightingSetting:new(),
		dof = DoFSetting:new(),
		wildAnimals = WildAnimalslSetting:new(),
		cameraCollisions = CameraCollisionsSetting:new(),
		guiCamera = GuiCameraSetting:new(),
		quitGame = QuitGameSetting:new(),
		clockColor = ClockColorSetting:new(),
		framerateLimiter = FramerateLimiterSetting:new(),
		clockBackground = ClockBackgroundSetting:new(),
		blinkingWarnings = BlinkingWarningsSetting:new()
	}
end

function AdditionalSettingsManager.addModEnvironmentTexts()
	for name, value in pairs(g_i18n.texts) do
		if string.startsWith(name, "additionalSettings_ui_") then
			g_env.g_i18n:setText(name, value)
		end
	end
end

function AdditionalSettingsManager.addSettingsPage(baseDirectory)
	local additionalSettingsFrame = AdditionalSettingsFrame.new()
	local additionalSettingsFrameReference = AdditionalSettingsFrameReference.new()

	local currentGui = FocusManager.currentGui

	g_gui:loadGui(Utils.getFilename("gui/AdditionalSettingsFrame.xml", baseDirectory), "AdditionalSettingsFrame", additionalSettingsFrame, true)
	g_gui:loadGui(Utils.getFilename("gui/AdditionalSettingsFrameReference.xml", baseDirectory), "AdditionalSettingsFrameReference", additionalSettingsFrameReference)

	if currentGui ~= nil then
		FocusManager:setGui(currentGui)
	end

	local inGameMenu = g_currentMission.inGameMenu
	local pageAdditionalSettings = additionalSettingsFrameReference.pageAdditionalSettings

	if pageAdditionalSettings ~= nil then
		local pagingElement = inGameMenu.pagingElement
		local index = pagingElement:getPageIndexByElement(inGameMenu.pageSettingsGeneral) + 1

		PagingElement:superClass().addElement(pagingElement, pageAdditionalSettings)
		pagingElement:addPage(string.upper(pageAdditionalSettings.name), pageAdditionalSettings, g_i18n:getText("additionalSettings_ui_ingameMenuAdditionalSettings"), index)

		inGameMenu:registerPage(pageAdditionalSettings, index, inGameMenu:makeIsGeneralSettingsEnabledPredicate())
		inGameMenu:addPageTab(pageAdditionalSettings, g_iconsUIFilename, GuiUtils.getUVs(SettingsScreen.TAB_UV.DEVICE_SETTINGS))
		inGameMenu.pageAdditionalSettings = pageAdditionalSettings
	end
end

function AdditionalSettingsManager:loadMap(filename)
	AdditionalSettingsUtil.appendedFunction(FSCareerMissionInfo, "saveToXMLFile", self, "saveToXMLFile")
	AdditionalSettingsUtil.raiseEvent("onLoad", filename)

	self.savedSettingStates = self:loadSettingsFromXMLFile()
	self:applySavedSettingStates(AdditionalSettingsManager.LOADING_STATE.LOAD_MAP)

	local pageAdditionalSettings = g_currentMission.inGameMenu.pageAdditionalSettings

	if pageAdditionalSettings ~= nil then
		pageAdditionalSettings:initialize(self)
	end
end

function AdditionalSettingsManager:onLoadMapFinished()
	AdditionalSettingsUtil.raiseEvent("onLoadFinished")
	self:applySavedSettingStates(AdditionalSettingsManager.LOADING_STATE.LOAD_MAP_FINISHED)
end

function AdditionalSettingsManager:onMissionStarted()
	AdditionalSettingsUtil.raiseEvent("onMissionStarted")
	self:applySavedSettingStates(AdditionalSettingsManager.LOADING_STATE.MISSION_START)

	self.savedSettingStates = nil
end

function AdditionalSettingsManager:deleteMap()
	AdditionalSettingsUtil.raiseEvent("onDelete")
end

function AdditionalSettingsManager:update(dt)
	AdditionalSettingsUtil.raiseEvent("onUpdate", dt)
end

function AdditionalSettingsManager:draw()
	AdditionalSettingsUtil.raiseEvent("onDraw")
end

function AdditionalSettingsManager:preDraw()
	if g_currentMission ~= nil and g_currentMission.isLoaded and g_currentMission.isRunning and not g_currentMission.hud:getIsFading() and g_gui.currentGuiName ~= "GamepadSigninScreen" and not g_gui:getIsGuiVisible() then
		AdditionalSettingsUtil.raiseEvent("onPreDraw")
	end
end

function AdditionalSettingsManager:getSettingByName(settingName)
	return self.addittionalSettings[settingName]
end

function AdditionalSettingsManager:getSettingStateByName(settingName)
	local setting = self:getSettingByName(settingName)

	if setting ~= nil then
		return setting.active or setting.state
	end

	return nil
end

function AdditionalSettingsManager.getUIElement(setting)
	local pageAdditionalSettings = g_currentMission.inGameMenu.pageAdditionalSettings

	if pageAdditionalSettings ~= nil then
		for checkboxElement, settingsKey in pairs(pageAdditionalSettings.checkboxMapping) do
			if settingsKey == setting then
				return checkboxElement
			end
		end

		for optionElement, settingsKey in pairs(pageAdditionalSettings.optionMapping) do
			if settingsKey == setting then
				return optionElement
			end
		end
	end

	return nil
end

function AdditionalSettingsManager.saveToXMLFile(settingsManager, object)
	settingsManager:saveSettingsToXMLFile()
end

function AdditionalSettingsManager:saveSettingsToXMLFile()
	local xmlFile = XMLFile.create("additionalSettingsXML", self.modSettings.xmlFilename, "additionalSettings")

	if xmlFile ~= nil then
		xmlFile:setString("additionalSettings#version", self.modVersion)

		for id, setting in pairs(self.addittionalSettings) do
			if setting.loadingState ~= AdditionalSettingsManager.LOADING_STATE.NO_LOAD then
				local key = "additionalSettings." .. id
				local customState = AdditionalSettingsUtil.callFunction(setting, "onSaveSetting", xmlFile, key)

				if not customState then
					if setting.active ~= nil then
						xmlFile:setBool(key, setting.active)
					elseif setting.state ~= nil then
						xmlFile:setInt(key, setting.state)
					end
				end
			end
		end

		xmlFile:save()
		xmlFile:delete()
	end
end

function AdditionalSettingsManager:loadSettingsFromXMLFile()
	local savedSettingStates = {}
	local xmlFile = XMLFile.loadIfExists("additionalSettingsXML", self.modSettings.xmlFilename, "additionalSettings")

	if xmlFile ~= nil then
		if self.resetAllowed and self.modVersion ~= xmlFile:getString("additionalSettings#version") then
			Logging.warning("[AdditionalSettings] mod update detected, settings will be reset.")
		else
			for id, setting in pairs(self.addittionalSettings) do
				local savedState
				local key = "additionalSettings." .. id
				local customState, savedState = AdditionalSettingsUtil.callFunction(setting, "onLoadSetting", xmlFile, key)

				if not customState then
					if setting.active ~= nil then
						local active = xmlFile:getBool(key)

						if active ~= nil and setting.active ~= active then
							savedState = active
						end
					elseif setting.state ~= nil then
						local state = xmlFile:getInt(key)

						if state ~= nil and setting.state ~= state then
							savedState = state
						end
					end
				end

				savedSettingStates[id] = savedState
			end
		end

		xmlFile:delete()
	end

	return savedSettingStates
end

function AdditionalSettingsManager:applySavedSettingStates(loadingState)
	for id, setting in pairs(self.addittionalSettings) do
		if setting.loadingState == loadingState then
			local savedSettingState = self.savedSettingStates[id]

			if savedSettingState ~= nil then
				if setting.active ~= nil and setting.active ~= savedSettingState then
					setting.active = savedSettingState
					AdditionalSettingsUtil.callFunction(setting, "onStateChange", savedSettingState, self.getUIElement(setting), true)
				elseif setting.state ~= nil and setting.state ~= savedSettingState then
					setting.state = savedSettingState
					AdditionalSettingsUtil.callFunction(setting, "onStateChange", savedSettingState, self.getUIElement(setting), true)
				end
			end
		end
	end
end